<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';

header('Content-Type: application/json');

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Function to log debug information
function debug_log($message, $data = null) {
    $log = date('Y-m-d H:i:s') . " - " . $message;
    if ($data !== null) {
        $log .= " - Data: " . print_r($data, true);
    }
    error_log($log);
}

// Test database connection
try {
    $pdo->query("SELECT 1");
    debug_log("Database connection successful");
} catch (PDOException $e) {
    debug_log("Database connection failed", $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => 'Database connection failed: ' . $e->getMessage()
    ]);
    exit();
}

// Check if products table exists
try {
    $stmt = $pdo->query("SHOW TABLES LIKE 'products'");
    if ($stmt->rowCount() === 0) {
        debug_log("Products table does not exist");
        echo json_encode([
            'success' => false,
            'message' => 'Products table does not exist'
        ]);
        exit();
    }
    debug_log("Products table exists");
} catch (PDOException $e) {
    debug_log("Error checking products table", $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => 'Error checking products table: ' . $e->getMessage()
    ]);
    exit();
}

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    debug_log("User not logged in");
    echo json_encode(['success' => false, 'message' => 'Not authorized']);
    exit();
}

// Get the search term
$term = isset($_GET['term']) ? trim($_GET['term']) : '';
debug_log("Search term received", $term);

if (empty($term)) {
    debug_log("No search term provided");
    echo json_encode(['success' => false, 'message' => 'No search term provided']);
    exit();
}

try {
    // First, let's check what columns exist in the products table
    $stmt = $pdo->query("DESCRIBE products");
    $columns = $stmt->fetchAll(PDO::FETCH_COLUMN);
    debug_log("Available columns in products table", $columns);

    // Search products by name or serial number
    $query = "
        SELECT p.id, p.name, 
            p.price AS selling_price, 
            p.cost_price,
            p.quantity AS stock_quantity,
            c.require_serials
        FROM products p
        LEFT JOIN categories c ON p.category_id = c.id
        LEFT JOIN product_serial_numbers psn ON p.id = psn.product_id
        WHERE p.name LIKE ? 
        OR psn.serial_number LIKE ?
        ORDER BY p.name 
        LIMIT 10
    ";
    
    $searchTerm = "%{$term}%";
    debug_log("Executing search query", [
        'query' => $query,
        'searchTerm' => $searchTerm
    ]);

    $stmt = $pdo->prepare($query);
    $stmt->execute([$searchTerm, $searchTerm]);
    $products = $stmt->fetchAll(PDO::FETCH_ASSOC);

    debug_log("Search results count", count($products));

    // Ensure all products have the expected keys
    $products = array_map(function($p) {
        return [
            'id' => $p['id'],
            'name' => $p['name'],
            'selling_price' => isset($p['selling_price']) ? $p['selling_price'] : (isset($p['price']) ? $p['price'] : 0),
            'cost_price' => isset($p['cost_price']) ? $p['cost_price'] : 0,
            'stock_quantity' => isset($p['stock_quantity']) ? $p['stock_quantity'] : (isset($p['quantity']) ? $p['quantity'] : 0),
            'require_serials' => isset($p['require_serials']) ? (int)$p['require_serials'] : 0
        ];
    }, $products);

    debug_log("Processed products", $products);

    echo json_encode([
        'success' => true,
        'products' => $products
    ]);
} catch (PDOException $e) {
    debug_log("Database error", $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
} 